define([
    'backbone',
    'marionette',
    'underscore',
    'App',
    'modules/appointments/appointments-radio',

    'text!modules/appointments/appointments-layout.html',
    'text!modules/page/content/_primary-header.html',

    'modules/appointments/booked/booked-appointments-view',
    'modules/appointments/requested/requested-appointments-view',
    'modules/page/content/aria-live-view',

    'modules/notifications/notifications-module',
    'modules/notifications/notifications-button-view',
],
function (
    Backbone,
    Marionette,
    _,
    app,
    radio,
    template,
    _primaryHeaderTemplate,
    BookedAppointmentsView,
    RequestedAppointmentsView,
    AriaLiveView,
    notificationsModule,
    NotificationsButtonView
) {
    'use strict';

    return Backbone.Marionette.View.extend({
        className: 'appointments',
        template: _.template(template),
        regions: {
            notificationsRegion: '.notifications-section',
            bookedAppointmentsRegion: '.appointments-section:nth-last-of-type(4)',
            requestedAppointmentsRegion: '.appointments-section:nth-last-of-type(3)',
            ariaLiveRegion: '.content-live-region',
        },
        events: {
            'click #refresh-appointments-btn': 'refreshAppointmentViews',
            'click #heading-new-appointment-request-btn': 'routeToNewAppointmentRequest',
            'click #new-appointment-request-btn': 'routeToNewAppointmentRequest',
            'click #notifications-button': 'routeToNotificationsPage',
        },

        onRender: function() {
            var modelToSet;
            var requestCollection = new Backbone.Collection();
            var appointmentsModule = this.module;
            $.when(appointmentsModule.requests.isReady).then(function() {
                appointmentsModule.requests.each(function(requestModel) {
                    if (requestModel.get('typeOfCareId') === 'CCPCPRVDR') {
                        modelToSet = _.find(appointmentsModule.appointments.models, function(model) {
                            return model.get('appointmentRequestId') === requestModel.get('appointmentRequestId');
                        });
                        requestModel.set('bookedAppointment', modelToSet);
                        appointmentsModule.appointments.remove(modelToSet);
                    }
                    requestCollection.add(requestModel);
                });
            });
            this.bookedAppointmentsView = new BookedAppointmentsView({
                id: 'booked-appointments',
                collection: this.module.appointments,
            });
            this.requestedAppointmentsView = new RequestedAppointmentsView({
                id: 'requested-appointments',
                collection: requestCollection,
            });
            this.onShow();
        },
        initialize: function(options) {
            this.googleAnalytics = options.googleAnalyticsProvider.getInstance();
            this.module = options.appointmentsModuleProvider.getInstance();
        },
        onShow: function() {
            $.when(notificationsModule.fetchCollection()).then(function() {
                this.showNotificationsButtonView();
            }.bind(this));
            $.when(this.module.appointments.isReady()).then(function() {
                this.showBookedAppointmentsView();
            }.bind(this));
            $.when(this.module.requests.isReady).then(function() {
                this.showRequestedAppointmentsView();
            }.bind(this));

            this.showAriaLiveView();
        },
        onDestroy: function() {
            delete this.bookedAppointmentsView;
            delete this.requestedAppointmentsView;
            delete this.ariaLiveView;
        },
        showNotificationsButtonView: function() {
            if (!this.isDestroyed()) {
                this.showChildView('notificationsRegion', new NotificationsButtonView({
                    collection: notificationsModule.notifications,
                }));
            }
        },
        showBookedAppointmentsView: function() {
            if (!this.isDestroyed()) {
                this.showChildView('bookedAppointmentsRegion', this.bookedAppointmentsView);
            }
        },
        showRequestedAppointmentsView: function() {
            if (!this.isDestroyed()) {
                this.showChildView('requestedAppointmentsRegion', this.requestedAppointmentsView);
            }
        },
        showAriaLiveView: function() {
            if (!this.isDestroyed() && !this.ariaLiveView) {
                this.ariaLiveView = new AriaLiveView({
                    model: new Backbone.Model({response: ''}),
                });

                this.showChildView('ariaLiveRegion', this.ariaLiveView);
            }
        },
        refreshAppointmentViews: function() {
            // eslint-disable-next-line consistent-this
            var appointmentsLayoutView = this;
            var _fetchCollectionsAndupdateAriaLiveView = function() {
                this.ariaLiveView.model.set('response', '');

                $.when(
                    this.module.appointments.fetchAppointmentData(),
                    this.module.requests.fetch({cache: false, reset: true})
                ).then(function () {
                    this.ariaLiveView.model.set('response', 'Appointments and Requests refreshed.');
                    appointmentsLayoutView.render();
                }.bind(this));
            }.bind(this);

            // Send an event when the user asks to refresh the appointments
            this.googleAnalytics.gas('send', 'event', 'veteran-appointment', 'refresh-appointments-clicked');
            

            if (!this.isDestroyed()) {
                _fetchCollectionsAndupdateAriaLiveView();
            }
        },
        routeToNewAppointmentRequest: function() {
            // when: CC_FEATURE_SET is disabled (e.g. its a defined property of disabledFeatuers)
            // then: go directly to new-va-appointment and skip CC appointment service question
            var ccIsDisabled = typeof app.disabledFeatures.get('CC_FEATURE_SET') !== 'undefined';
            var IsEligibleForCC = app.ccEligibilityModel.isCcEligible();
            var ccEligFeatureEnabled = app.isCcEligFeatureEnabled();
            var ccRequestingIsDisabled = ccEligFeatureEnabled === true && IsEligibleForCC === false;
            if (ccIsDisabled === true) {
                Backbone.history.navigate('new-va-appointment-request', {trigger: true});
            } else if (ccRequestingIsDisabled === true) {
                Backbone.history.navigate('new-va-appointment-request', {trigger: true});
            } else {
                Backbone.history.navigate('new-appointment-request', {trigger: true});
            }

            if (typeof gas !== 'undefined') {
                gas('send', 'event', 'veteran-appointment', 'new-appointment-request-clicked');
            }
        },
        routeToNotificationsPage: function() {
            Backbone.history.navigate('notifications', {trigger: true});
        },
        templateContext: {
            primaryHeader: _.template(_primaryHeaderTemplate),
        },
    });
}
);
